<?php
// selenium-scraper-auto-bypass.php
require __DIR__ . '/vendor/autoload.php';

use Facebook\WebDriver\Remote\RemoteWebDriver;
use Facebook\WebDriver\Remote\DesiredCapabilities;
use Facebook\WebDriver\Chrome\ChromeOptions;
use Facebook\WebDriver\WebDriverBy;
use Facebook\WebDriver\Interactions\WebDriverActions;

const BASE_URL     = 'https://www.sahibinden.com';
const LIST_PATH    = '/emlak?address_country=270';
const LINK_CACHE   = __DIR__ . '/detailLinks.json';
const OUTPUT_FILE  = __DIR__ . '/test_agents.json';
const MAX_AGENTS   = 3;

/**
 * Her run’da taze bir profil klasörü oluştur
 */
function createFreshProfile(): string {
    $dir = __DIR__ . '/profiles/profile-' . time();
    mkdir($dir, 0777, true);
    return $dir;
}

/**
 * ChromeDriver + Selenium başlat, otomasyon izlerini sakıncalı flag’lerle etkisizleştir
 */
function startDriver(string $profileDir): RemoteWebDriver {
    $opts = new ChromeOptions();
    $opts->addArguments([
        "--user-data-dir={$profileDir}",
        '--disable-blink-features=AutomationControlled',
        '--disable-infobars',
        '--window-size=1200,900',
        // gerçek bir masaüstü UA taklidi
        '--user-agent=Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 '.
                         '(KHTML, like Gecko) Chrome/114.0.0.0 Safari/537.36',
    ]);
    // isteğe bağlı: web güvenliğini gevşet
    $opts->addArguments(['--disable-web-security','--ignore-certificate-errors']);

    $caps = DesiredCapabilities::chrome();
    $caps->setCapability(ChromeOptions::CAPABILITY, $opts);
    return RemoteWebDriver::create('http://localhost:4444/wd/hub', $caps, 15000);
}

/** Fareyi rastgele hareket ettirir */
function humanMoveMouse(RemoteWebDriver $d) {
    try {
        $dim = $d->executeScript('return {w:window.innerWidth,h:window.innerHeight};');
        $a = new WebDriverActions($d);
        $a->moveByOffset(random_int(10,$dim['w']-20), random_int(10,$dim['h']-20))->perform();
        usleep(random_int(200000,500000));
    } catch (\Exception){}
}

/** Sayfayı aşağı kaydırır */
function humanScroll(RemoteWebDriver $d) {
    $h = $d->executeScript('return document.body.scrollHeight;');
    $step = max(100,intval($h/5));
    for ($y=0;$y<=$h;$y+=$step) {
        $d->executeScript("window.scrollTo(0,{$y});");
        usleep(random_int(300000,700000));
    }
    sleep(1);
}

// 1) Cache’den ilk 10 ilan detay linkini al
if (!file_exists(LINK_CACHE)) {
    exit("Error: detailLinks.json bulunamadı. Önce linkleri cache’e alın.\n");
}
$detailLinks = array_slice(json_decode(file_get_contents(LINK_CACHE), true), 0, MAX_AGENTS);
echo "➤ İlk " . count($detailLinks) . " ilan linki test edilecek.\n";

// 2) Profil ve driver hazırla
$profileDir = createFreshProfile();
$driver     = startDriver($profileDir);

/**
 * İlan detayından acente sayfası URL’si çıkarır.
 * Login/redirection görürse doğrudan null döner.
 */
function extractAgentPage(RemoteWebDriver $d, string $detailUrl): ?string {
    $d->get($detailUrl);
    sleep(2);

    // eğer login sayfasına yönlendiyse atla
    if (strpos($d->getCurrentURL(), '/giris') !== false) {
        echo "  ⚠️ Redirect /giris tespit edildi, atlanıyor\n";
        return null;
    }

    try {
        $agentName = trim(
            $d->findElement(WebDriverBy::xpath(
                "//*[@id='classifiedDetail']//div[contains(@class,'classifiedContactSeller')]//h3"
            ))->getText()
        );
        $xp = sprintf(
            "//a[@title=%s and normalize-space(text())=%s]",
            json_encode($agentName), json_encode($agentName)
        );
        return $d->findElement(WebDriverBy::xpath($xp))
                 ->getAttribute('href');
    } catch (\Exception) {
        // fallback: ilan detay içinde başka bir link algıla
        try {
            $el = $d->findElement(WebDriverBy::cssSelector(
                "a[href*='sahibinden.com/emlak-']"
            ));
            return $el->getAttribute('href');
        } catch (\Exception) {
            return null;
        }
    }
}

/**
 * Acente sayfasından bilgileri ve ekip üyelerini çeker.
 * Yine login/redirection kontrolü ekledik.
 */
function parseAgentPage(RemoteWebDriver $d, string $agentUrl): array {
    $d->get($agentUrl);
    sleep(2);
    if (strpos($d->getCurrentURL(), '/giris') !== false) {
        echo "  ⚠️ Acente sayfası /giris’e yönlendi, atlanıyor\n";
        return ['agencyName'=>'','location'=>'','team'=>[]];
    }
    try {
        $agencyName = trim(
            $d->findElement(WebDriverBy::xpath(
                "//*[@id='container']//div[@class='store-name']/preceding-sibling::h1"
            ))->getText()
        );
    } catch (\Exception) { $agencyName = ''; }

    try {
        $location = $d->findElement(WebDriverBy::xpath(
            "//div[@class='store-name']"
        ))->getAttribute('data-main-location');
    } catch (\Exception) { $location = ''; }

    // Ekip
    $d->get(rtrim($agentUrl, '/') . '/ekibimiz');
    sleep(2);
    $team = [];
    $cards = $d->findElements(WebDriverBy::xpath(
        "//*[@id='container']/div[1]/div[3]/div/div"
    ));
    foreach ($cards as $card) {
        try { $name  = trim($card->findElement(WebDriverBy::cssSelector("p.name"))->getText()); }
        catch (\Exception){ continue; /* isim yoksa skip */ }
        try { $tag   = trim($card->findElement(
                WebDriverBy::xpath(".//p[normalize-space(.)!='{$name}']")
            )->getText()); }
        catch (\Exception){ $tag = ''; }
        try { $phone = $card->findElement(
                WebDriverBy::cssSelector("a.phone-trigger")
            )->getAttribute('data-phone'); }
        catch (\Exception){ $phone = ''; }
        $team[] = ['name'=>$name,'tag'=>$tag,'phone'=>$phone];
    }

    return ['agencyName'=>$agencyName,'location'=>$location,'team'=>$team];
}

// 3) Her ilan linki için acente verilerini çek
$results = [];
foreach ($detailLinks as $detailUrl) {
    echo "▶ {$detailUrl}\n";
    humanMoveMouse($driver);
    humanScroll($driver);

    $agentUrl = extractAgentPage($driver, $detailUrl);
    if (!$agentUrl) {
        echo "  ⚠️ Acente linki bulunamadı veya atlandı.\n";
        continue;
    }
    echo "  → {$agentUrl}\n";

    $info = parseAgentPage($driver, $agentUrl);
    $results[] = array_merge(['agentUrl'=>$agentUrl], $info);
}

$driver->quit();
file_put_contents(OUTPUT_FILE,
    json_encode($results, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE)
);
echo "✅ Sonuçlar " . OUTPUT_FILE . " dosyasına yazıldı.\n";
