<?php
require __DIR__ . '/vendor/autoload.php';

use Symfony\Component\BrowserKit\HttpBrowser;
use Symfony\Component\HttpClient\HttpClient;
use Google\Client;
use Google\Service\Sheets;
use Google\Service\Sheets\ValueRange;

// ––––––––––––––––––––––––––––––––
// 1) Google Sheets API Setup
// ––––––––––––––––––––––––––––––––
$google = new Client();
$google->setApplicationName('KKTC Emlak Acente Scraper');
$google->setScopes([Sheets::SPREADSHEETS]);
$google->setAuthConfig(__DIR__ . '/kktc-emlak-scraper.json');

$sheetsService  = new Sheets($google);
$spreadsheetId  = '1eUlKD7eHTcKYNHyNngox-j20FDRdcF7O_buk7Pzv8A8';
$sheetName      = 'Sahibinden';

// Write headers
$header = new ValueRange([
    'values' => [[
        'Acente Adı',
        'Konum',
        'Ekibimiz – Üye 1',
        'Ekibimiz – Üye 2',
        'Ekibimiz – Üye 3',
    ]]
]);
$sheetsService->spreadsheets_values->update(
    $spreadsheetId,
    "{$sheetName}!A1",
    $header,
    ['valueInputOption' => 'RAW']
);

// ––––––––––––––––––––––––––––––––
// 2) Puppeteer: fetch detail‐page URLs
// ––––––––––––––––––––––––––––––––
echo "▶ Spawning Puppeteer to collect detail links…\n";
$json = shell_exec('node ' . escapeshellarg(__DIR__ . '/puppeteer-fetch.js'));
$detailLinks = json_decode($json, true);
if (!is_array($detailLinks)) {
    exit("⚠️  Failed to load detail links from Puppeteer helper\n");
}
echo "➤ Collected " . count($detailLinks) . " detail links\n";

// ––––––––––––––––––––––––––––––––
// 3) HTTP Browser for parsing pages
// ––––––––––––––––––––––––––––––––
$client     = HttpClient::create([
    'timeout'       => 15,
    'headers'       => ['User-Agent' => 'Mozilla/5.0'],
    'max_redirects' => 5,
]);
$browser    = new HttpBrowser($client);

// Extract agent page URL from a listing detail
function extractAgentPage(HttpBrowser $browser, string $detailUrl): ?string {
    $crawler = $browser->request('GET', $detailUrl);
    $node    = $crawler
        ->filterXPath('//a[contains(@href,".sahibinden.com/emlak")]')
        ->getNode(0);
    if ($node instanceof \DOMElement && $node->hasAttribute('href')) {
        $href = $node->getAttribute('href');
        if (str_starts_with($href, '//')) {
            return 'https:' . $href;
        }
        if (str_starts_with($href, '/')) {
            return 'https://www.sahibinden.com' . $href;
        }
        return $href;
    }
    return null;
}

// Parse agent page for details
function parseAgentPage(HttpBrowser $browser, string $agentUrl): array {
    $crawler = $browser->request('GET', $agentUrl);
    $xp      = fn($xp) => trim($crawler->filterXPath($xp)->text() ?? '');

    return [
        $xp("//*[@id='container']/div[1]/div[2]/div[2]/div[3]/div[1]"),  // Name
        $xp("//*[@id='container']/div[1]/div[2]/div[2]/div[3]/div[1]"),  // Location (adjust if needed)
        $xp("//*[@id='container']/div[1]/div[3]/div/div[1]/div[1]/div/div[2]/p[1]"), // Team1
        $xp("//*[@id='container']/div[1]/div[3]/div/div[2]/div[1]/div/div[2]/p[1]"), // Team2
        $xp("//*[@id='container']/div[1]/div[3]/div/div[3]/div[1]/div/div[2]/p[1]"), // Team3
    ];
}

// ––––––––––––––––––––––––––––––––
// 4) Collect unique agent pages
// ––––––––––––––––––––––––––––––––
$agentPages = [];
foreach ($detailLinks as $detailUrl) {
    echo "▶ Detail: {$detailUrl}\n";
    $agentUrl = extractAgentPage($browser, $detailUrl);
    if ($agentUrl) {
        $agentPages[$agentUrl] = true;
        echo "  → Agent page: {$agentUrl}\n";
    }
    usleep(200_000);
}
$agentPages = array_keys($agentPages);
echo "➤ Found " . count($agentPages) . " unique agent pages\n";

// ––––––––––––––––––––––––––––––––
// 5) Parse each agent page
// ––––––––––––––––––––––––––––––––
$rows = [];
foreach ($agentPages as $au) {
    echo "▶ Parsing agent: {$au}\n";
    $rows[] = parseAgentPage($browser, $au);
    usleep(200_000);
}

// ––––––––––––––––––––––––––––––––
// 6) Append to Google Sheets
// ––––––––––––––––––––––––––––––––
if (!empty($rows)) {
    $body = new ValueRange(['values' => $rows]);
    $sheetsService->spreadsheets_values->append(
        $spreadsheetId,
        "{$sheetName}!A2",
        $body,
        [
          'valueInputOption' => 'RAW',
          'insertDataOption' => 'INSERT_ROWS'
        ]
    );
    echo "✅ " . count($rows) . " agent records written to Sheets\n";
} else {
    echo "⚠️ No agent data found\n";
}
