// scrape-agents.js
require('dotenv').config();
const fs        = require('fs');
const path      = require('path');
const os        = require('os');
const mysql     = require('mysql2/promise');
const readline  = require('readline');
const puppeteer = require('puppeteer-extra');
const Stealth   = require('puppeteer-extra-plugin-stealth');
puppeteer.use(Stealth());

// ——— Sabitler ———
const SOURCE_NAME       = 'Sahibinden';
const DB_CONFIG         = { host:'localhost', user:'root', password:'', database:'emlak_db', charset:'utf8mb4' };
const DETAIL_FILE       = path.resolve(__dirname, 'pendingLinks.json');
const PROXY_FILE        = path.resolve(__dirname, 'proxies.txt');
const PROFILE_ROOT      = path.resolve(__dirname, 'profiles');
const MAX_PROXY_TRIES   = 5;   // Proxy deneme hakkı
const MAX_DETAIL_TRIES  = 3;   // Detay sayfa deneme hakkı
const CLOUD_URL         = 'https://www.sahibinden.com';

// ——— Yardımcılar ———
if (!fs.existsSync(PROFILE_ROOT)) fs.mkdirSync(PROFILE_ROOT, { recursive:true });

function pruneOldProfiles(maxAgeMinutes) {
  const now = Date.now(), maxAgeMs = maxAgeMinutes*60_000;
  for (let name of fs.readdirSync(PROFILE_ROOT)) {
    const full = path.join(PROFILE_ROOT,name);
    if (fs.statSync(full).isDirectory() && now - fs.statSync(full).mtimeMs > maxAgeMs) {
      console.log(`🗑️ Siliniyor (yaş>${maxAgeMinutes}dk): ${name}`);
      fs.rmSync(full,{recursive:true,force:true});
    }
  }
}

function createProfile() {
  const sub = `profile-${Date.now()}`, dir = path.join(PROFILE_ROOT,sub);
  fs.mkdirSync(dir,{recursive:true});
  return dir;
}

const humanDelay = (min=500,max=2500)=> new Promise(r=>setTimeout(r,min+Math.random()*(max-min)));

function randomUserAgent(){
  const UAS = [
    'Mozilla/5.0 (Windows NT 10.0; Win64; x64)...',
    'Mozilla/5.0 (Macintosh; Intel Mac OS X 13_4)...',
    'Mozilla/5.0 (X11; Linux x86_64)...'
  ];
  return UAS[Math.floor(Math.random()*UAS.length)];
}

function randomViewport(){
  return {
    width:  Math.floor(1024+Math.random()*416),
    height: Math.floor(700 +Math.random()*200)
  };
}

async function humanScroll(page){
  await page.evaluate(()=>window.scrollTo(0,Math.random()*document.body.scrollHeight/3));
  await humanDelay(300,800);
  await page.evaluate(()=>window.scrollTo(0,document.body.scrollHeight));
  await humanDelay(300,800);
}

async function humanMoveMouse(page){
  const vp=page.viewport(), x=Math.random()*(vp.width-20), y=Math.random()*(vp.height-20);
  await page.mouse.move(x,y); await humanDelay(200,500);
}

function waitForEnter(txt='') {
  return new Promise(r=>readline.createInterface({input:process.stdin,output:process.stdout})
    .question(txt,()=>r()));
}

// ——— Proxy Listesi ———
let PROXIES = [];
if (fs.existsSync(PROXY_FILE)) {
  PROXIES = fs.readFileSync(PROXY_FILE,'utf8')
    .trim().split('\n')
    .map(l=>{ const [h,p,u,pa]=l.split(':'); return {server:`${h}:${p}`, user:u, pass:pa}; });
}
let proxyIdx=0;
function nextProxy() {
  if (!PROXIES.length) return null;
  return PROXIES[ proxyIdx++ % PROXIES.length ];
}

// ——— Tarayıcı Başlatma ———
async function launchBrowser(proxy){
  const profileDir = createProfile();
  console.log('🔑 Profil:',profileDir);
  const args = [
    `--user-data-dir=${profileDir}`,
    '--disable-blink-features=AutomationControlled',
    '--no-sandbox','--disable-setuid-sandbox'
  ];
  if (proxy) args.push(`--proxy-server=${proxy.server}`);
  return puppeteer.launch({
    headless:false,
    executablePath: os.platform()==='win32'
      ? 'C:/Program Files/Google/Chrome/Application/chrome.exe'
      : undefined,
    args, defaultViewport:null, slowMo:50
  });
}

// ——— Ana Akış ———
;(async()=>{
  pruneOldProfiles(60);

  const conn = await mysql.createConnection(DB_CONFIG);
  console.log('✅ MySQL’e bağlanıldı.');

  if (!fs.existsSync(DETAIL_FILE)) {
    console.error('Error: pendingLinks.json yok.');
    process.exit(1);
  }
  const detailLinks = JSON.parse(fs.readFileSync(DETAIL_FILE,'utf8'));

  for (let detailUrl of detailLinks) {
    console.log('\n▶ Detay:',detailUrl);

    let page, browser, proxy;
    // Proxy + Cloudflare + detay retry
    for (let attempt=1; attempt<=MAX_PROXY_TRIES; attempt++) {
      proxy = nextProxy();
      console.log(`🔄 Proxy#${attempt}: ${proxy?.server}${proxy?.user?` (auth:${proxy.user})`:''}`);
      browser = await launchBrowser(proxy);
      page    = (await browser.pages())[0] || await browser.newPage();
      if (proxy?.user) await page.authenticate({username:proxy.user,password:proxy.pass});
      await page.setUserAgent(randomUserAgent());
      await page.setViewport(randomViewport());

      // Anasayfa Cloudflare
      try { await page.goto(CLOUD_URL,{waitUntil:'networkidle2',timeout:60000}); }
      catch {}
      await humanDelay(1000,2000);
      if (page.url().includes('secure.sahibinden.com')) {
        console.log('⚠️ Cloudflare onayı: ENTER→');
        await waitForEnter();
      }

      // Detay sayfa
      try {
        await page.goto(detailUrl,{waitUntil:'networkidle2',timeout:30000});
      } catch(e) {
        console.warn(`  ⚠️ Detay yüklenemedi (try#${attempt}), proxy değiştiriliyor…`);
        await browser.close();
        continue;
      }
      // eğer Cloudflare tekrar gelmişse
      if (page.url().includes('secure.sahibinden.com')) {
        console.warn(`  ⚠️ Cloudflare devreye girdi (try#${attempt}), proxy değiştiriliyor…`);
        await browser.close();
        continue;
      }
      break;
    }
    if (!page || page.url() !== detailUrl) {
      console.error('  🚫 Atlanıyor.');
      continue;
    }

    // insanlaştırma
    await humanDelay(500,1500);
    await humanScroll(page);
    await humanMoveMouse(page);

    // 4) Acente linki
    let agentHref = await page.$eval('div.user-info-store-name>a',a=>a.href).catch(()=>null);
    if (!agentHref) {
      agentHref = await page.$eval('#classifiedDetail div[class*=classifiedContactSeller] a',a=>a.href).catch(()=>null);
    }
    if (!agentHref) {
      console.log('  ⚠️ Acente linki bulunamadı.');
      await browser.close();
      continue;
    }
    console.log('  → Acente:', agentHref);

    // 5) Acenta sayfa + Cloudflare retry
    for (let a=1; a<=MAX_DETAIL_TRIES; a++){
      try {
        await page.goto(agentHref,{waitUntil:'networkidle2',timeout:30000});
      } catch {
        console.warn(`  ⚠️ Acenta yüklenemedi (try#${a}), yeniden dene…`);
        continue;
      }
      if (page.url().includes('secure.sahibinden.com')) {
        console.warn(`  ⚠️ Acenta CF (try#${a}), bekle ENTER→`);
        await waitForEnter();
        continue;
      }
      break;
    }
    await humanDelay(800,1500);

    const storeDiv = await page.$('div.contact-info div.store-name');
    const agencyName = storeDiv ? await storeDiv.evaluate(e=>e.textContent.trim()) : '';
    const location   = storeDiv ? await storeDiv.evaluate(e=>e.getAttribute('data-main-location')) : '';
    console.log(`  → ${agencyName} — ${location}`);

    // 6) DB agents
    const [ins] = await conn.execute(
      `INSERT INTO agents(source,agency_name,agent_url,location)
       VALUES(?,?,?,?)
       ON DUPLICATE KEY UPDATE agency_name=VALUES(agency_name),location=VALUES(location)`,
      [SOURCE_NAME,agencyName,agentHref,location]
    );
    let agentId = ins.insertId;
    if (!agentId) {
      const [[r]] = await conn.execute(`SELECT id FROM agents WHERE agent_url=?`,[agentHref]);
      agentId = r.id;
    }

    // 7) Ekibimiz
    const {origin} = new URL(agentHref), teamUrl=`${origin}/ekibimiz`;
    console.log('  ▶ Ekibimiz:',teamUrl);
    try {
      await page.goto(teamUrl,{waitUntil:'networkidle2',timeout:30000});
    } catch {
      console.warn('  ⚠️ Ekibimiz yüklenemedi, atlanıyor…');
      await browser.close();
      continue;
    }
    await humanDelay(800,1500);

    // 8) üyeler
    const members = await page.$$eval('div.edr-trigger.consultant',cards=>cards.map(c=>({
      name:c.querySelector('p.name')?.innerText.trim()||'',
      tag:c.querySelector('p.title')?.innerText.trim()||'',
      phone:c.querySelector('a.phone-trigger')?.getAttribute('data-phone')||''
    }))).catch(()=>[]);
    for (const m of members) {
      await conn.execute(
        `INSERT INTO team_members(agent_id,name,tag,phone)
         VALUES(?,?,?,?)
         ON DUPLICATE KEY UPDATE phone=VALUES(phone)`,
        [agentId,m.name,m.tag,m.phone]
      );
    }
    console.log(`  ✅ ${members.length} ekip üyesi kaydedildi.`);

    await browser.close();
  }

  await conn.end();
  console.log('\n🎉 İşlem tamamlandı.');
})();
