// controller.js

const { execSync } = require('child_process');
const fs           = require('fs');
const path         = require('path');

// 1) Ayarları oku
const configPath = path.resolve(__dirname, 'config.json');
if (!fs.existsSync(configPath)) {
  console.error('Error: config.json bulunamadı.');
  process.exit(1);
}
const { maxNewLinksPerRun, runIntervalMinutes } = JSON.parse(fs.readFileSync(configPath, 'utf8'));

console.log(`ℹ️ Ayarlar: maxNewLinksPerRun=${maxNewLinksPerRun}, runIntervalMinutes=${runIntervalMinutes}`);

// Ana işlevi bir fonksiyona alalım
function runOnce() {
  console.log('\n▶ ===== Controller çalıştırılıyor =====');

  // A) İşlenmiş linkler
  const processedPath = path.resolve(__dirname, 'processedLinks.json');
  let processed = [];
  if (fs.existsSync(processedPath)) {
    processed = JSON.parse(fs.readFileSync(processedPath, 'utf8'));
  }

  // B) DetailLinks güncelle
  console.log('▶ generate_detail_links.js çalıştırılıyor…');
  execSync('node generate_detail_links.js', { stdio: 'inherit' });

  // C) Tüm linkler
  const detailPath = path.resolve(__dirname, 'detailLinks.json');
  const allLinks   = JSON.parse(fs.readFileSync(detailPath, 'utf8'));

  // D) Yeni linkleri süz
  const newLinks = allLinks.filter(l => !processed.includes(l));
  if (newLinks.length === 0) {
    console.log('ℹ️ Yeni link yok, bu döngüde hiçbir şey yapılmayacak.');
  } else {
    // E) Sınırlama uygula
    const toProcess = newLinks.slice(0, maxNewLinksPerRun);
    console.log(`✅ ${toProcess.length} yeni link seçildi (toplam ${newLinks.length}).`);

    // F) Geçici pending dosyası
    const pendingPath = path.resolve(__dirname, 'pendingLinks.json');
    fs.writeFileSync(pendingPath, JSON.stringify(toProcess, null, 2), 'utf8');

    // G) scrape-agents çalıştır
    console.log('▶ scrape-agents.js çalıştırılıyor…');
    execSync(`node scrape-agents.js pendingLinks.json`, { stdio: 'inherit' });

    // H) processedLinks.json güncelle
    processed = processed.concat(toProcess);
    fs.writeFileSync(processedPath, JSON.stringify(processed, null, 2), 'utf8');
    console.log(`🎉 processedLinks.json güncellendi (artık ${processed.length} link).`);
  }
}

function scheduleNextRun() {
  const baseMs = runIntervalMinutes * 60 * 1000;
  // -%10 … +%10 arasında rastgele sapma
  const jitter = baseMs * 0.1;
  const delay = baseMs + (Math.random() * 2 - 1) * jitter;
  console.log(`🔄 Bir sonraki çalıştırma yaklaşık ${(delay/60000).toFixed(1)} dakikaya ayarlandı.`);
  setTimeout(() => {
    runOnce();
    scheduleNextRun();
  }, delay);
}
// controller.js içindeki runOnce()’dan önce:
const { profileMaxAgeMinutes } = JSON.parse(fs.readFileSync(configPath, 'utf8'));
const profileRoot = path.resolve(__dirname, 'profiles');

// prune fonksiyonu
function pruneOldProfiles(rootDir, maxAgeMinutes) {
  if (!fs.existsSync(rootDir)) return;
  const now = Date.now();
  const maxAgeMs = maxAgeMinutes * 60 * 1000;
  for (const name of fs.readdirSync(rootDir)) {
    const full = path.join(rootDir, name);
    const stat = fs.statSync(full);
    if (stat.isDirectory() && now - stat.mtimeMs > maxAgeMs) {
      console.log(`🗑️ Siliniyor (yaş >${maxAgeMinutes}dk): ${name}`);
      fs.rmSync(full, { recursive: true, force: true });
    }
  }
}

// runOnce()’ın en başına ekle:
pruneOldProfiles(profileRoot, profileMaxAgeMinutes);

// 2) İlk çalıştırma
runOnce();

// 3) Rastgele aralıkla tekrarla
if (runIntervalMinutes > 0) {
  scheduleNextRun();
}